#!/usr/bin/env php
<?php
/**
 * Master Automation Scheduler for GalliumHash Ecosystem
 * 
 * This script coordinates automated data collection and intelligence processing
 * cPanel Optimized: Respects hosting limits, uses fallbacks, includes error handling
 * 
 * CRON SETUP:
 * Add to cPanel cron jobs:
 * */10 * * * * /usr/bin/php /path/to/core.galliumhash.com/automation/master-scheduler.php
 */

// Set timezone
date_default_timezone_set('UTC');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Paths
$projectRoot = __DIR__;
$logFile = $projectRoot . '/automation/logs/master-scheduler.log';

// Create logs directory
if (!is_dir(dirname($logFile))) {
    mkdir(dirname($logFile), 0755, true);
}

/**
 * Simple logging function
 */
function logMessage($message, $level = 'INFO') {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

/**
 * Execute command safely with timeout
 */
function executeCommand($command, $timeout = 300) {
    $output = [];
    $returnCode = 0;
    
    try {
        $process = proc_open(
            $command,
            [
                0 => ['pipe', 'r'],
                1 => ['pipe', 'w'], 
                2 => ['pipe', 'w']
            ],
            $pipes
        );
        
        if (is_resource($process)) {
            fclose($pipes[0]);
            
            // Wait with timeout
            $startTime = time();
            while (!feof($pipes[1])) {
                $output[] = fgets($pipes[1]);
                
                if (time() - $startTime > $timeout) {
                    proc_terminate($process);
                    return [
                        'success' => false,
                        'output' => implode('', $output),
                        'error' => 'Timeout exceeded'
                    ];
                }
            }
            
            fclose($pipes[1]);
            fclose($pipes[2]);
            $returnCode = proc_close($process);
        }
        
        return [
            'success' => $returnCode === 0,
            'output' => implode('', $output),
            'return_code' => $returnCode
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'output' => '',
            'error' => $e->getMessage()
        ];
    }
}

/**
 * Main automation logic
 */
function runAutomationCycle() {
    global $projectRoot;
    
    logMessage("Starting automation cycle");
    
    $results = [
        'metabuilder' => null,
        'gas_collector' => null,
        'ethereum_miner' => null
    ];
    
    // 1. MetaBuilder Data Collection (every 10 minutes)
    logMessage("Running MetaBuilder data collection...");
    $result = executeCommand("cd $projectRoot && php metabuilder/metabuilder.php --action=collect", 120);
    $results['metabuilder'] = $result;
    
    if ($result['success']) {
        logMessage("✅ MetaBuilder collection completed successfully");
    } else {
        logMessage("❌ MetaBuilder collection failed: " . ($result['error'] ?? 'Unknown error'), 'ERROR');
    }
    
    // 2. Ethereum GasCollector (every 5 minutes)
    logMessage("Running Ethereum GasCollector...");
    $result = executeCommand("cd $projectRoot/metachain/ethereum/builder/collectors && php GasCollector.php", 60);
    $results['gas_collector'] = $result;
    
    if ($result['success']) {
        logMessage("✅ GasCollector completed successfully");
    } else {
        logMessage("❌ GasCollector failed: " . ($result['error'] ?? 'Unknown error'), 'ERROR');
    }
    
    // 3. MetaMiner Intelligence Processing (every 5 minutes)
    logMessage("Running MetaMiner intelligence processing...");
    $result = executeCommand("cd $projectRoot && php metaminer/oracle-api.php", 90);
    $results['ethereum_miner'] = $result;
    
    if ($result['success']) {
        logMessage("✅ MetaMiner processing completed successfully");
    } else {
        logMessage("❌ MetaMiner processing failed: " . ($result['error'] ?? 'Unknown error'), 'ERROR');
    }
    
    // Summary
    $successCount = 0;
    foreach ($results as $system => $result) {
        if ($result && $result['success']) {
            $successCount++;
        }
    }
    
    logMessage("Automation cycle completed. Success: $successCount/3 systems");
    
    return [
        'timestamp' => date('c'),
        'systems_processed' => $results,
        'success_count' => $successCount,
        'total_systems' => 3
    ];
}

/**
 * Health check for all systems
 */
function checkSystemHealth() {
    global $projectRoot;
    
    $health = [
        'timestamp' => date('c'),
        'systems' => []
    ];
    
    // Check MetaBuilder
    try {
        $result = executeCommand("cd $projectRoot && php metabuilder/metabuilder.php --action=health", 30);
        $health['systems']['metabuilder'] = $result['success'] ? 'healthy' : 'unhealthy';
    } catch (Exception $e) {
        $health['systems']['metabuilder'] = 'error';
    }
    
    // Check MetaMiner
    try {
        $result = executeCommand("cd $projectRoot && php metaminer/oracle-api.php", 30);
        $health['systems']['metaminer'] = $result['success'] ? 'healthy' : 'unhealthy';
    } catch (Exception $e) {
        $health['systems']['metaminer'] = 'error';
    }
    
    // Check database connection
    try {
        $result = executeCommand("cd $projectRoot && php -r \"require 'metabuilder/utils/Database.php'; echo Database::getInstance() ? 'connected' : 'disconnected';\"", 10);
        $health['systems']['database'] = strpos($result['output'], 'connected') !== false ? 'healthy' : 'unhealthy';
    } catch (Exception $e) {
        $health['systems']['database'] = 'error';
    }
    
    return $health;
}

// Main execution
try {
    $action = $argv[1] ?? 'run';
    
    switch ($action) {
        case 'run':
            $results = runAutomationCycle();
            echo json_encode($results, JSON_PRETTY_PRINT) . "\n";
            break;
            
        case 'health':
            $health = checkSystemHealth();
            echo json_encode($health, JSON_PRETTY_PRINT) . "\n";
            break;
            
        case 'test':
            logMessage("Test execution mode - checking system availability");
            $health = checkSystemHealth();
            echo "System Health Check:\n";
            foreach ($health['systems'] as $system => $status) {
                echo "- $system: $status\n";
            }
            break;
            
        default:
            echo "Usage: php master-scheduler.php [run|health|test]\n";
            echo "  run    - Run full automation cycle\n";
            echo "  health - Check system health\n";
            echo "  test   - Test system availability\n";
    }
    
} catch (Exception $e) {
    logMessage("Fatal error: " . $e->getMessage(), 'ERROR');
    echo json_encode([
        'error' => true,
        'message' => $e->getMessage(),
        'timestamp' => date('c')
    ], JSON_PRETTY_PRINT) . "\n";
    exit(1);
}