#!/usr/bin/env php
<?php
/**
 * Complete System Test for GalliumHash Ecosystem
 * 
 * Tests all automation and security fixes
 * Verifies that all systems work after our enhancements
 */

date_default_timezone_set('UTC');

echo "🧪 GALLIUMHASH ECOSYSTEM - COMPLETE SYSTEM TEST\n";
echo "================================================\n\n";

$projectRoot = __DIR__ . '/..';
$testResults = [
    'environment_variables' => false,
    'config_files' => false,
    'automation_scripts' => false,
    'database_connection' => false,
    'metabuilder_functionality' => false,
    'metaminer_functionality' => false,
    'ethereum_functionality' => false,
    'overall_status' => 'unknown'
];

/**
 * Test environment variables
 */
function testEnvironmentVariables() {
    echo "🔧 Testing Environment Variables...\n";
    
    $envFile = __DIR__ . '/../../.env';
    
    if (!file_exists($envFile)) {
        echo "  ❌ .env file not found\n";
        return false;
    }
    
    echo "  ✅ .env file exists\n";
    
    // Load environment variables
    $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $variablesFound = 0;
    
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && !empty(trim($line)) && !str_starts_with($line, '#')) {
            list($key, $value) = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value);
            $variablesFound++;
        }
    }
    
    echo "  ✅ Loaded $variablesFound environment variables\n";
    
    // Check critical variables
    $criticalVars = ['DB_PASSWORD', 'FLASHBOTS_API_KEY', 'ETHERSCAN_API_KEY'];
    foreach ($criticalVars as $var) {
        if (isset($_ENV[$var]) && !empty($_ENV[$var])) {
            echo "  ✅ $var configured\n";
        } else {
            echo "  ❌ $var missing or empty\n";
            return false;
        }
    }
    
    return true;
}

/**
 * Test config files
 */
function testConfigFiles() {
    echo "\n📝 Testing Config Files...\n";
    
    $configFiles = [
        '/metaminer/config.php',
        '/metabuilder/config/config.php',
        '/wit/config.php',
        '/metachain/ethereum/builder/collectors/GasCollector.php'
    ];
    
    foreach ($configFiles as $configFile) {
        $fullPath = __DIR__ . '/..' . $configFile;
        
        if (!file_exists($fullPath)) {
            echo "  ❌ Config file missing: $configFile\n";
            return false;
        }
        
        echo "  ✅ Config file exists: $configFile\n";
        
        // Check if file uses environment variables
        $content = file_get_contents($fullPath);
        if (strpos($content, '$_ENV') !== false) {
            echo "  ✅ Uses environment variables: $configFile\n";
        } else {
            echo "  ❌ Does not use environment variables: $configFile\n";
            return false;
        }
    }
    
    return true;
}

/**
 * Test automation scripts
 */
function testAutomationScripts() {
    echo "\n🤖 Testing Automation Scripts...\n";
    
    $scripts = [
        'automation/master-scheduler.php',
        'automation/metabuilder-collector.php',
        'automation/metaminer-processor.php',
        'automation/ethereum-gas-collector.php',
        'automation/system-health.php'
    ];
    
    foreach ($scripts as $script) {
        $fullPath = __DIR__ . '/..' . '/' . $script;
        
        if (!file_exists($fullPath)) {
            echo "  ❌ Automation script missing: $script\n";
            return false;
        }
        
        echo "  ✅ Automation script exists: $script\n";
        
        // Check file permissions
        if (is_readable($fullPath)) {
            echo "  ✅ Script is readable: $script\n";
        } else {
            echo "  ❌ Script is not readable: $script\n";
            return false;
        }
    }
    
    return true;
}

/**
 * Test database connection
 */
function testDatabaseConnection() {
    echo "\n🗄️ Testing Database Connection...\n";
    
    try {
        // Test MetaBuilder database
        $dsn = "mysql:host=" . ($_ENV['DB_HOST'] ?? 'localhost') . 
               ";dbname=" . ($_ENV['DB_NAME'] ?? 'nvdaxcom_galliumhash_metabuilder') . 
               ";charset=utf8mb4";
        $username = $_ENV['DB_USERNAME'] ?? 'nvdaxcom_galliumhash_metabuilder_user';
        $password = $_ENV['DB_PASSWORD'] ?? '32TbgulnE7YOd2G0e5';
        
        $db = new PDO($dsn, $username, $password);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        echo "  ✅ MetaBuilder database connection successful\n";
        
        // Test with MetaMiner config
        require_once __DIR__ . '/../metaminer/config.php';
        $config = require __DIR__ . '/../metaminer/config.php';
        
        $minerDb = new PDO(
            "mysql:host={$config['database']['host']};dbname={$config['database']['database']};charset=utf8mb4",
            $config['database']['username'],
            $config['database']['password']
        );
        $minerDb->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        echo "  ✅ MetaMiner database connection successful\n";
        
        return true;
        
    } catch (Exception $e) {
        echo "  ❌ Database connection failed: " . $e->getMessage() . "\n";
        return false;
    }
}

/**
 * Test MetaBuilder functionality
 */
function testMetaBuilderFunctionality() {
    echo "\n🏗️ Testing MetaBuilder Functionality...\n";
    
    try {
        chdir(__DIR__ . '/../metabuilder');
        
        // Test health check
        $output = [];
        exec("php metabuilder.php --action=health 2>&1", $output, $returnCode);
        
        if ($returnCode === 0) {
            echo "  ✅ MetaBuilder health check successful\n";
            return true;
        } else {
            echo "  ❌ MetaBuilder health check failed\n";
            echo "  Error: " . implode("\n", $output) . "\n";
            return false;
        }
        
    } catch (Exception $e) {
        echo "  ❌ MetaBuilder test failed: " . $e->getMessage() . "\n";
        return false;
    }
}

/**
 * Test MetaMiner functionality
 */
function testMetaMinerFunctionality() {
    echo "\n⚡ Testing MetaMiner Functionality...\n";
    
    try {
        chdir(__DIR__ . '/../metaminer');
        
        // Test oracle API
        $output = [];
        exec("php oracle-api.php 2>&1", $output, $returnCode);
        
        if ($returnCode === 0) {
            echo "  ✅ MetaMiner oracle API successful\n";
            
            // Try to parse response
            $response = implode("\n", $output);
            $data = json_decode($response, true);
            
            if ($data && isset($data['data_freshness'])) {
                echo "  ✅ MetaMiner returns valid intelligence data\n";
            }
            
            return true;
        } else {
            echo "  ❌ MetaMiner oracle API failed\n";
            return false;
        }
        
    } catch (Exception $e) {
        echo "  ❌ MetaMiner test failed: " . $e->getMessage() . "\n";
        return false;
    }
}

/**
 * Test Ethereum functionality
 */
function testEthereumFunctionality() {
    echo "\n⛽ Testing Ethereum Functionality...\n";
    
    try {
        chdir(__DIR__ . '/../metachain/ethereum/builder/collectors');
        
        // Test GasCollector instantiation
        $testCode = '
        require_once "GasCollector.php";
        $collector = new GasCollector();
        echo "GasCollector instantiated successfully\n";
        exit(0);
        ';
        
        file_put_contents('/tmp/gas_test.php', $testCode);
        
        $output = [];
        exec("php /tmp/gas_test.php 2>&1", $output, $returnCode);
        
        unlink('/tmp/gas_test.php');
        
        if ($returnCode === 0 && strpos(implode('', $output), 'success') !== false) {
            echo "  ✅ Ethereum GasCollector functional\n";
            return true;
        } else {
            echo "  ❌ Ethereum GasCollector test failed\n";
            echo "  Output: " . implode("\n", $output) . "\n";
            return false;
        }
        
    } catch (Exception $e) {
        echo "  ❌ Ethereum test failed: " . $e->getMessage() . "\n";
        return false;
    }
}

/**
 * Main test execution
 */
try {
    $testResults['environment_variables'] = testEnvironmentVariables();
    $testResults['config_files'] = testConfigFiles();
    $testResults['automation_scripts'] = testAutomationScripts();
    $testResults['database_connection'] = testDatabaseConnection();
    $testResults['metabuilder_functionality'] = testMetaBuilderFunctionality();
    $testResults['metaminer_functionality'] = testMetaMinerFunctionality();
    $testResults['ethereum_functionality'] = testEthereumFunctionality();
    
    // Determine overall status
    $successfulTests = array_filter($testResults, function($result) {
        return $result === true;
    });
    
    $testResults['overall_status'] = count($successfulTests) >= 6 ? 'excellent' : 
                                    (count($successfulTests) >= 4 ? 'good' : 'needs_attention');
    
    echo "\n" . str_repeat("=", 50) . "\n";
    echo "🎯 TEST RESULTS SUMMARY\n";
    echo str_repeat("=", 50) . "\n";
    
    foreach ($testResults as $test => $result) {
        if ($test === 'overall_status') continue;
        
        $status = $result ? '✅ PASS' : '❌ FAIL';
        echo "$test: $status\n";
    }
    
    echo "\nOverall Status: " . strtoupper($testResults['overall_status']) . "\n";
    
    if ($testResults['overall_status'] === 'excellent') {
        echo "\n🎉 EXCELLENT! All systems are working perfectly.\n";
        echo "Your GalliumHash ecosystem is ready for automation!\n\n";
        
        echo "📋 NEXT STEPS:\n";
        echo "1. Set up cron jobs using the guide in automation/CRON_SETUP.md\n";
        echo "2. Monitor system health at /automation/system-health.php\n";
        echo "3. Check logs in /automation/logs/ for any issues\n";
        echo "4. Your systems will now run automatically! 🚀\n";
        
    } elseif ($testResults['overall_status'] === 'good') {
        echo "\n⚠️ GOOD! Most systems are working.\n";
        echo "Please review the failed tests and fix any issues.\n";
        
    } else {
        echo "\n🚨 NEEDS ATTENTION! Several systems have issues.\n";
        echo "Please fix the failed tests before proceeding with automation.\n";
    }
    
    echo "\n📊 Test completed at: " . date('Y-m-d H:i:s T') . "\n";
    
    exit($testResults['overall_status'] === 'excellent' ? 0 : 1);
    
} catch (Exception $e) {
    echo "\n❌ Fatal error during testing: " . $e->getMessage() . "\n";
    exit(1);
}