-- =====================================================
-- MetaChain Database Setup
-- Database: nvdaxcom_galliumhash_metachain
-- Purpose: Multi-chain intelligence and universal mining
-- Created: 2025-10-24
-- =====================================================

-- Create database if not exists
CREATE DATABASE IF NOT EXISTS nvdaxcom_galliumhash_metachain CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE nvdaxcom_galliumhash_metachain;

-- =====================================================
-- USER MANAGEMENT TABLES
-- =====================================================

-- Core user profiles for multi-chain system
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    wallet_address VARCHAR(255),
    total_tokens DECIMAL(20,8) DEFAULT 0.00000000,
    tier VARCHAR(20) DEFAULT 'casual', -- casual, engaged, power, builder
    verification_level INT DEFAULT 0, -- 0=basic, 1=verified, 2=premium
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_username (username),
    INDEX idx_email (email),
    INDEX idx_wallet (wallet_address),
    INDEX idx_tier (tier)
);

-- User verification and KYC
CREATE TABLE user_verifications (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    verification_type VARCHAR(50) NOT NULL, -- email, phone, kyc, wallet
    status VARCHAR(20) DEFAULT 'pending', -- pending, verified, rejected
    data JSON, -- Store verification data
    verified_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_verification (user_id, verification_type)
);

-- =====================================================
-- CHAIN MANAGEMENT TABLES
-- =====================================================

-- Supported blockchain networks
CREATE TABLE chains (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(50) NOT NULL,
    symbol VARCHAR(10) NOT NULL,
    rpc_url TEXT,
    explorer_url TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_chain_name (name),
    UNIQUE KEY unique_chain_symbol (symbol)
);

-- Insert default chains
INSERT INTO chains (name, symbol, rpc_url, explorer_url) VALUES
('Bitcoin', 'BTC', 'https://api.bitcoinnode.com', 'https://blockstream.info'),
('Ethereum', 'ETH', 'https://eth-mainnet.alchemyapi.io', 'https://etherscan.io'),
('Solana', 'SOL', 'https://api.mainnet-beta.solana.com', 'https://explorer.solana.com'),
('Polygon', 'MATIC', 'https://polygon-rpc.com', 'https://polygonscan.com'),
('Arbitrum', 'ARB', 'https://arb1.arbitrum.io/rpc', 'https://arbiscan.io'),
('Optimism', 'OP', 'https://mainnet.optimism.io', 'https://optimistic.etherscan.io'),
('Base', 'BASE', 'https://mainnet.base.org', 'https://basescan.org'),
('BNB Chain', 'BNB', 'https://bsc-dataseed.binance.org', 'https://bscscan.com');

-- =====================================================
-- UNIVERSAL MINING METHODS TABLES
-- =====================================================

-- 8 Universal mining methods tracking
CREATE TABLE mining_methods (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(50) NOT NULL,
    description TEXT,
    token_rate DECIMAL(10,6) DEFAULT 0.000000, -- Tokens earned per action
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_method_name (name)
);

-- Insert universal mining methods
INSERT INTO mining_methods (name, description, token_rate) VALUES
('prediction_mining', 'Earn tokens by making accurate market predictions', 5.000000),
('price_tracking_mining', 'Engage with price alerts and tracking activities', 2.500000),
('opportunity_scouting_mining', 'Hunt for arbitrage and alpha opportunities', 10.000000),
('social_intelligence_mining', 'Provide community intelligence and insights', 7.500000),
('cross_chain_arbitrage_mining', 'Identify and report cross-chain price differences', 15.000000),
('educational_mining', 'Create educational content and tutorials', 3.000000),
('data_labeling_mining', 'Contribute to AI training datasets', 4.000000),
('community_moderation_mining', 'Moderate content and maintain community standards', 6.000000);

-- User mining activities tracking
CREATE TABLE user_mining_activities (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    mining_method_id INT NOT NULL,
    activity_type VARCHAR(100) NOT NULL,
    chain_id INT,
    metadata JSON, -- Store method-specific data
    tokens_earned DECIMAL(10,6) DEFAULT 0.000000,
    status VARCHAR(20) DEFAULT 'pending', -- pending, approved, rejected
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    approved_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (mining_method_id) REFERENCES mining_methods(id),
    FOREIGN KEY (chain_id) REFERENCES chains(id),
    INDEX idx_user_activity (user_id),
    INDEX idx_method_activity (mining_method_id),
    INDEX idx_chain_activity (chain_id),
    INDEX idx_status (status)
);

-- =====================================================
-- PREDICTION MARKETS TABLES
-- =====================================================

-- Prediction markets for each chain
CREATE TABLE prediction_markets (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    chain_id INT NOT NULL,
    prediction_type VARCHAR(50), -- price, volume, tvl, transactions, etc.
    question TEXT NOT NULL,
    options JSON NOT NULL, -- Store prediction options
    start_time TIMESTAMP NOT NULL,
    end_time TIMESTAMP NOT NULL,
    status VARCHAR(20) DEFAULT 'active', -- active, closed, resolved
    total_participants INT DEFAULT 0,
    total_stake DECIMAL(20,8) DEFAULT 0.00000000,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (chain_id) REFERENCES chains(id),
    INDEX idx_chain_prediction (chain_id),
    INDEX idx_status_time (status, end_time)
);

-- User predictions
CREATE TABLE user_predictions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    market_id INT NOT NULL,
    selected_option VARCHAR(100) NOT NULL,
    stake_amount DECIMAL(20,8) NOT NULL,
    tokens_at_stake DECIMAL(10,6) NOT NULL,
    prediction_confidence DECIMAL(3,2), -- 0.00 to 1.00
    reasoning TEXT,
    status VARCHAR(20) DEFAULT 'active', -- active, won, lost, cancelled
    payout_amount DECIMAL(20,8) DEFAULT 0.00000000,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    resolved_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (market_id) REFERENCES prediction_markets(id),
    INDEX idx_user_predictions (user_id),
    INDEX idx_market_predictions (market_id),
    INDEX idx_status (status)
);

-- Prediction market results
CREATE TABLE prediction_results (
    id INT PRIMARY KEY AUTO_INCREMENT,
    market_id INT NOT NULL,
    winning_option VARCHAR(100) NOT NULL,
    actual_value DECIMAL(20,8),
    resolved_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (market_id) REFERENCES prediction_markets(id),
    UNIQUE KEY unique_market_result (market_id)
);

-- =====================================================
-- TOKEN ECONOMY TABLES
-- =====================================================

-- Token transactions for all activities
CREATE TABLE token_transactions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    transaction_type VARCHAR(50) NOT NULL, -- mining, prediction_win, stake, unstake, transfer, bonus
    amount DECIMAL(20,8) NOT NULL,
    description VARCHAR(255),
    related_table VARCHAR(50), -- Which table this transaction relates to
    related_id INT, -- ID in related table
    balance_before DECIMAL(20,8) NOT NULL,
    balance_after DECIMAL(20,8) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    INDEX idx_user_transactions (user_id),
    INDEX idx_transaction_type (transaction_type),
    INDEX idx_created_at (created_at)
);

-- Token earning rules and bonuses
CREATE TABLE earning_rules (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    rule_type VARCHAR(50) NOT NULL, -- daily_bonus, streak_bonus, achievement_bonus
    trigger_conditions JSON, -- Conditions that trigger this bonus
    bonus_amount DECIMAL(10,6) DEFAULT 0.000000,
    multiplier DECIMAL(5,2) DEFAULT 1.00,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_rule_type (rule_type)
);

-- =====================================================
-- INTELLIGENCE HUBS TABLES
-- =====================================================

-- Ethereum-specific intelligence data
CREATE TABLE ethereum_intelligence (
    id INT PRIMARY KEY AUTO_INCREMENT,
    gas_price DECIMAL(10,2),
    base_fee DECIMAL(10,2),
    priority_fee DECIMAL(10,2),
    block_number BIGINT,
    mev_activity DECIMAL(20,8),
    staking_tvl DECIMAL(30,8),
    validator_count INT,
    l2_comparisons JSON,
    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_timestamp (timestamp),
    INDEX idx_block_number (block_number)
);

-- Solana-specific intelligence data
CREATE TABLE solana_intelligence (
    id INT PRIMARY KEY AUTO_INCREMENT,
    slot_number BIGINT,
    tps DECIMAL(10,2),
    network_congestion DECIMAL(5,2),
    validator_count INT,
    total_staked DECIMAL(30,8),
    avg_transaction_fee DECIMAL(20,8),
    defi_tvl DECIMAL(30,8),
    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_timestamp (timestamp),
    INDEX idx_slot_number (slot_number)
);

-- Cross-chain analytics and arbitrage opportunities
CREATE TABLE cross_chain_analytics (
    id INT PRIMARY KEY AUTO_INCREMENT,
    chain1_id INT NOT NULL,
    chain2_id INT NOT NULL,
    asset_symbol VARCHAR(10) NOT NULL,
    price_chain1 DECIMAL(20,8) NOT NULL,
    price_chain2 DECIMAL(20,8) NOT NULL,
    price_difference_percentage DECIMAL(10,4),
    arbitrage_opportunity BOOLEAN DEFAULT FALSE,
    volume_analysis JSON,
    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (chain1_id) REFERENCES chains(id),
    FOREIGN KEY (chain2_id) REFERENCES chains(id),
    INDEX idx_chains (chain1_id, chain2_id),
    INDEX idx_timestamp (timestamp),
    INDEX idx_arbitrage (arbitrage_opportunity)
);

-- DeFi protocol tracking
CREATE TABLE defi_protocols (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    chain_id INT NOT NULL,
    protocol_type VARCHAR(50), -- lending, dex, yield, derivatives
    tvl DECIMAL(30,8),
    apy DECIMAL(8,4),
    volume_24h DECIMAL(30,8),
    token_price DECIMAL(20,8),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (chain_id) REFERENCES chains(id),
    INDEX idx_chain_protocol (chain_id, protocol_type),
    INDEX idx_tvl (tvl),
    INDEX idx_active (is_active)
);

-- NFT marketplace tracking
CREATE TABLE nft_markets (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    chain_id INT NOT NULL,
    floor_price DECIMAL(20,8),
    volume_24h DECIMAL(30,8),
    sales_count_24h INT,
    average_price DECIMAL(20,8),
    unique_buyers_24h INT,
    trending_collections JSON,
    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (chain_id) REFERENCES chains(id),
    INDEX idx_chain_market (chain_id),
    INDEX idx_timestamp (timestamp)
);

-- =====================================================
-- LEADERBOARD SYSTEM TABLES
-- =====================================================

-- Global leaderboard rankings
CREATE TABLE global_leaderboard (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    total_tokens DECIMAL(20,8) NOT NULL,
    mining_streak_days INT DEFAULT 0,
    prediction_accuracy DECIMAL(5,2) DEFAULT 0.00,
    global_rank INT,
    rank_updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    UNIQUE KEY unique_user_global_rank (user_id),
    INDEX idx_global_rank (global_rank),
    INDEX idx_total_tokens (total_tokens)
);

-- Chain-specific leaderboards
CREATE TABLE chain_leaderboards (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    chain_id INT NOT NULL,
    chain_tokens DECIMAL(20,8) NOT NULL,
    chain_predictions INT DEFAULT 0,
    chain_mining_activities INT DEFAULT 0,
    chain_rank INT,
    rank_updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (chain_id) REFERENCES chains(id),
    UNIQUE KEY unique_user_chain_rank (user_id, chain_id),
    INDEX idx_chain_rank (chain_id, chain_rank),
    INDEX idx_chain_tokens (chain_tokens)
);

-- Method-specific leaderboards
CREATE TABLE method_leaderboards (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    mining_method_id INT NOT NULL,
    method_tokens DECIMAL(20,8) NOT NULL,
    method_activities INT DEFAULT 0,
    method_rank INT,
    rank_updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (mining_method_id) REFERENCES mining_methods(id),
    UNIQUE KEY unique_user_method_rank (user_id, mining_method_id),
    INDEX idx_method_rank (mining_method_id, method_rank),
    INDEX idx_method_tokens (method_tokens)
);

-- =====================================================
-- ACHIEVEMENT SYSTEM TABLES
-- =====================================================

-- Available achievements
CREATE TABLE achievements (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    badge_icon VARCHAR(100),
    category VARCHAR(50), -- mining, prediction, social, learning
    token_reward DECIMAL(10,6) DEFAULT 0.000000,
    conditions JSON, -- Conditions to unlock achievement
    rarity VARCHAR(20) DEFAULT 'common', -- common, rare, epic, legendary
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_category (category),
    INDEX idx_rarity (rarity)
);

-- User achievements
CREATE TABLE user_achievements (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    achievement_id INT NOT NULL,
    unlocked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    bonus_claimed BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (achievement_id) REFERENCES achievements(id),
    UNIQUE KEY unique_user_achievement (user_id, achievement_id),
    INDEX idx_user_achievements (user_id)
);

-- =====================================================
-- CACHE AND PERFORMANCE TABLES
-- =====================================================

-- API response caching
CREATE TABLE api_cache (
    id INT PRIMARY KEY AUTO_INCREMENT,
    cache_key VARCHAR(255) UNIQUE NOT NULL,
    cache_data LONGTEXT NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_cache_key (cache_key),
    INDEX idx_expires_at (expires_at)
);

-- System settings and configurations
CREATE TABLE system_settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT NOT NULL,
    description TEXT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_setting_key (setting_key)
);

-- Insert default system settings
INSERT INTO system_settings (setting_key, setting_value, description) VALUES
('token_symbol', 'GHM', 'GalliumHash Meta token symbol'),
('mining_rewards_multiplier', '1.0', 'Global multiplier for mining rewards'),
('prediction_win_rate_bonus', '2.0', 'Win rate multiplier for token rewards'),
('max_daily_predictions', '50', 'Maximum predictions per user per day'),
('cross_chain_arbitrage_bonus', '5.0', 'Bonus multiplier for cross-chain activities'),
('achievement_refresh_interval', '3600', 'Achievement checking interval in seconds'),
('leaderboard_update_interval', '1800', 'Leaderboard update interval in seconds'),
('api_cache_ttl', '300', 'API cache time-to-live in seconds');

-- =====================================================
-- TRIGGERS FOR AUTOMATION
-- =====================================================

-- Trigger to update user tier based on total tokens
DELIMITER //
CREATE TRIGGER update_user_tier_after_transaction
AFTER INSERT ON token_transactions
FOR EACH ROW
BEGIN
    DECLARE user_total DECIMAL(20,8);
    DECLARE new_tier VARCHAR(20);
    
    -- Calculate new total tokens for user
    SELECT total_tokens INTO user_total
    FROM users WHERE id = NEW.user_id;
    
    -- Determine new tier based on total tokens
    IF user_total >= 1000.00000000 THEN
        SET new_tier = 'builder';
    ELSEIF user_total >= 200.00000000 THEN
        SET new_tier = 'power';
    ELSEIF user_total >= 50.00000000 THEN
        SET new_tier = 'engaged';
    ELSE
        SET new_tier = 'casual';
    END IF;
    
    -- Update user tier if changed
    UPDATE users 
    SET tier = new_tier, updated_at = CURRENT_TIMESTAMP 
    WHERE id = NEW.user_id AND tier != new_tier;
END//
DELIMITER ;

-- =====================================================
-- VIEWS FOR ANALYTICS
-- =====================================================

-- View for user activity summary
CREATE VIEW user_activity_summary AS
SELECT 
    u.id as user_id,
    u.username,
    u.tier,
    u.total_tokens,
    COUNT(DISTINCT uma.id) as total_activities,
    COUNT(DISTINCT up.id) as total_predictions,
    AVG(up.prediction_confidence) as avg_prediction_confidence,
    COALESCE(SUM(uma.tokens_earned), 0) as total_mining_earnings,
    COALESCE(SUM(CASE WHEN up.status = 'won' THEN up.payout_amount ELSE 0 END), 0) as total_prediction_winnings,
    MAX(uma.created_at) as last_activity
FROM users u
LEFT JOIN user_mining_activities uma ON u.id = uma.user_id
LEFT JOIN user_predictions up ON u.id = up.user_id
GROUP BY u.id, u.username, u.tier, u.total_tokens;

-- View for chain analytics summary
CREATE VIEW chain_analytics_summary AS
SELECT 
    c.name as chain_name,
    c.symbol as chain_symbol,
    COUNT(DISTINCT uma.user_id) as active_users,
    SUM(uma.tokens_earned) as total_mining_tokens,
    COUNT(DISTINCT up.id) as total_predictions,
    COALESCE(AVG(prediction_accuracy.accuracy), 0) as avg_prediction_accuracy,
    MAX(uma.created_at) as last_activity
FROM chains c
LEFT JOIN user_mining_activities uma ON c.id = uma.chain_id
LEFT JOIN user_predictions up ON c.id = (SELECT chain_id FROM prediction_markets WHERE id = up.market_id)
LEFT JOIN (
    SELECT market_id, 
           SUM(CASE WHEN status = 'won' THEN 1 ELSE 0 END) * 100.0 / COUNT(*) as accuracy
    FROM user_predictions 
    GROUP BY market_id
) prediction_accuracy ON up.market_id = prediction_accuracy.market_id
WHERE c.is_active = TRUE
GROUP BY c.id, c.name, c.symbol;

-- =====================================================
-- INITIAL DATA SETUP
-- =====================================================

-- Insert default achievements
INSERT INTO achievements (name, description, badge_icon, category, token_reward, conditions, rarity) VALUES
('First Miner', 'Complete your first mining activity', '⛏️', 'mining', 10.000000, '{"activities": 1}', 'common'),
('Prediction Master', 'Make 10 accurate predictions', '🎯', 'prediction', 50.000000, '{"accurate_predictions": 10}', 'rare'),
('Cross-Chain Explorer', 'Participate in activities across 3 different chains', '🔗', 'mining', 25.000000, '{"different_chains": 3}', 'epic'),
('Social Intelligence', 'Provide 50 social intelligence contributions', '🧠', 'social', 30.000000, '{"social_contributions": 50}', 'rare'),
('Token Millionaire', 'Earn 1000 total tokens', '💰', 'mining', 100.000000, '{"total_tokens": 1000}', 'legendary'),
('DeFi Detective', 'Complete 25 DeFi-related mining activities', '🔍', 'mining', 40.000000, '{"defi_activities": 25}', 'epic'),
('NFT Curator', 'Contribute to 20 NFT market insights', '🎨', 'mining', 35.000000, '{"nft_contributions": 20}', 'rare'),
('Arbitrage Hunter', 'Identify 10 profitable cross-chain arbitrage opportunities', '⚡', 'mining', 75.000000, '{"arbitrage_opportunities": 10}', 'legendary');

-- Insert default earning rules
INSERT INTO earning_rules (name, description, rule_type, trigger_conditions, bonus_amount) VALUES
('Daily Login Bonus', 'Tokens for daily platform engagement', 'daily_bonus', '{"action": "daily_login"}', 5.000000),
('Mining Streak Bonus', 'Bonus for consecutive days of mining', 'streak_bonus', '{"consecutive_days": 7}', 25.000000),
('Prediction Accuracy Bonus', 'Bonus for high prediction accuracy', 'achievement_bonus', '{"accuracy_threshold": 0.8}', 15.000000),
('Cross-Chain Pioneer', 'First to use cross-chain features', 'achievement_bonus', '{"cross_chain_first": true}', 50.000000),
('Early Adopter Bonus', 'Bonus for being among first 1000 users', 'achievement_bonus', '{"user_rank": 1000}', 100.000000);

-- =====================================================
-- INDEXES FOR PERFORMANCE
-- =====================================================

-- Additional performance indexes
CREATE INDEX idx_token_transactions_user_created ON token_transactions(user_id, created_at);
CREATE INDEX idx_mining_activities_method_created ON user_mining_activities(mining_method_id, created_at);
CREATE INDEX idx_user_predictions_market_created ON user_predictions(market_id, created_at);
CREATE INDEX idx_prediction_markets_status_end ON prediction_markets(status, end_time);
CREATE INDEX idx_cross_chain_analytics_opportunity ON cross_chain_analytics(arbitrage_opportunity, timestamp);
CREATE INDEX idx_ethereum_intelligence_timestamp ON ethereum_intelligence(timestamp);
CREATE INDEX idx_solana_intelligence_timestamp ON solana_intelligence(timestamp);

-- =====================================================
-- COMMENTS AND DOCUMENTATION
-- =====================================================

/*
MetaChain Database Schema v1.0
===============================

This database schema supports the Multi-Chain Intelligence Center with:

1. USER MANAGEMENT:
   - Multi-chain user profiles with wallet integration
   - Verification system for different levels
   - Token balance tracking with tier system

2. UNIVERSAL MINING:
   - 8 universal mining methods tracking
   - Activity-based token earning system
   - Method-specific leaderboards

3. PREDICTION MARKETS:
   - Chain-specific prediction markets
   - User predictions with confidence scoring
   - Automated result resolution and payouts

4. TOKEN ECONOMY:
   - Complete transaction history
   - Earning rules and bonus system
   - Tier progression automation

5. INTELLIGENCE HUBS:
   - Ethereum intelligence data
   - Solana intelligence data
   - Cross-chain arbitrage tracking
   - DeFi and NFT market analytics

6. LEADERBOARD SYSTEM:
   - Global rankings
   - Chain-specific rankings
   - Method-specific rankings

7. ACHIEVEMENT SYSTEM:
   - Badge-based achievements
   - Token rewards for milestones
   - User progress tracking

Database Performance:
- Optimized indexes for common queries
- Automated triggers for tier updates
- Caching system for API responses
- Views for analytics and reporting

Ready for production deployment with multi-chain scalability.
*/