<?php
/**
 * Ethereum Gas Prophet - Advanced Gas Price Prediction Intelligence
 * Purpose: Predict optimal transaction timing and cost optimization
 * Analysis: Time-series, pattern recognition, event correlation
 * Output: Strategic recommendations with confidence scoring
 */

class GasProphet {
    private $db;
    private $eth_price_usd;
    private $prediction_confidence;
    
    public function __construct() {
        $this->db = new PDO("mysql:host=localhost;dbname=nvdaxcom_galliumhash_metachain;charset=utf8mb4", 
                           "nvdaxcom_galliumhash_metabuilder_user", 
                           "32TbgulnE7YOd2G0e5");
        $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $this->eth_price_usd = $this->getEthPrice();
    }
    
    /**
     * Get current ETH price from database or API
     */
    private function getEthPrice() {
        try {
            $stmt = $this->db->query("SELECT eth_price_usd FROM user_activity_summary LIMIT 1");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($result && $result['eth_price_usd']) {
                return floatval($result['eth_price_usd']);
            }
        } catch (Exception $e) {
            // Fallback price if database query fails
            return 2300.0;
        }
    }
    
    /**
     * Analyze gas patterns using advanced time-series analysis
     */
    private function analyzeGasPatterns() {
        $blocks = $this->getHistoricalGasData(7200); // Last 24 hours (7200 blocks)
        
        if (empty($blocks)) {
            throw new Exception("No gas data available for analysis");
        }
        
        $gas_prices = array_column($blocks, 'base_fee');
        $timestamps = array_column($blocks, 'timestamp');
        
        // Calculate moving averages
        $ma_1h = $this->calculateMovingAverage($gas_prices, 300); // 300 blocks (1 hour)
        $ma_6h = $this->calculateMovingAverage($gas_prices, 1800); // 1800 blocks (6 hours)
        $ma_24h = $this->calculateMovingAverage($gas_prices, 7200); // 7200 blocks (24 hours)
        $ma_7d = $this->calculateMovingAverage($gas_prices, 50400); // 50400 blocks (7 days)
        
        $current_gas = end($gas_prices);
        $current_ma_24h = end($ma_24h);
        
        // Calculate deviation from mean
        $deviation = (($current_gas - $current_ma_24h) / $current_ma_24h) * 100;
        
        // Determine gas state
        $gas_state = $this->classifyGasState($deviation);
        
        // Calculate volatility
        $volatility = $this->calculateVolatility($gas_prices);
        
        return [
            'current_gas' => $current_gas,
            'ma_1h' => end($ma_1h),
            'ma_6h' => end($ma_6h),
            'ma_24h' => end($ma_24h),
            'ma_7d' => end($ma_7d),
            'deviation_percent' => $deviation,
            'gas_state' => $gas_state,
            'volatility' => $volatility,
            'data_points' => count($gas_prices)
        ];
    }
    
    /**
     * Perform hourly pattern recognition analysis
     */
    private function analyzeHourlyPatterns() {
        $blocks = $this->getHistoricalGasData(50400); // 7 days
        
        $hourly_averages = [];
        for ($hour = 0; $hour < 24; $hour++) {
            $hour_gas_prices = [];
            
            foreach ($blocks as $block) {
                $block_hour = date('H', strtotime($block['timestamp']));
                if ($block_hour == sprintf('%02d', $hour)) {
                    $hour_gas_prices[] = $block['base_fee'];
                }
            }
            
            $hourly_averages[$hour] = [
                'average' => !empty($hour_gas_prices) ? array_sum($hour_gas_prices) / count($hour_gas_prices) : 0,
                'count' => count($hour_gas_prices),
                'std_dev' => !empty($hour_gas_prices) ? $this->calculateStdDev($hour_gas_prices) : 0
            ];
        }
        
        // Find patterns
        $cheapest_hour = array_search(min(array_column($hourly_averages, 'average')), array_column($hourly_averages, 'average'));
        $most_expensive_hour = array_search(max(array_column($hourly_averages, 'average')), array_column($hourly_averages, 'average'));
        
        $current_hour = intval(date('H'));
        $expected_gas = $hourly_averages[$current_hour]['average'];
        $current_gas = end($this->getRecentGasPrices(100));
        
        $anomaly_detected = abs($current_gas - $expected_gas) > (2 * $hourly_averages[$current_hour]['std_dev']);
        
        return [
            'hourly_patterns' => $hourly_averages,
            'cheapest_hour' => $cheapest_hour,
            'most_expensive_hour' => $most_expensive_hour,
            'current_hour_expected_gas' => $expected_gas,
            'current_gas' => $current_gas,
            'anomaly_detected' => $anomaly_detected
        ];
    }
    
    /**
     * Perform regression analysis for trend prediction
     */
    private function analyzeTrends() {
        $blocks = $this->getRecentGasPrices(100);
        $gas_prices = array_column($blocks, 'base_fee');
        $block_numbers = array_column($blocks, 'block_number');
        
        // Perform linear regression
        $regression = $this->performLinearRegression($block_numbers, $gas_prices);
        
        $slope = $regression['slope'];
        $intercept = $regression['intercept'];
        $r_squared = $regression['r_squared'];
        
        // Classify trend
        if ($slope > 0.5) {
            $trend = 'RISING';
        } elseif ($slope < -0.5) {
            $trend = 'FALLING';
        } else {
            $trend = 'STABLE';
        }
        
        // Predict next hour (300 blocks ahead)
        $current_block = end($block_numbers);
        $predicted_gas_1h = $slope * ($current_block + 300) + $intercept;
        
        // Predict next 6 hours (1800 blocks)
        $predicted_gas_6h = $slope * ($current_block + 1800) + $intercept;
        
        return [
            'trend' => $trend,
            'slope' => $slope,
            'r_squared' => $r_squared,
            'current_gas' => end($gas_prices),
            'predicted_gas_1h' => max(0, $predicted_gas_1h), // Ensure non-negative
            'predicted_gas_6h' => max(0, $predicted_gas_6h),
            'confidence' => min(100, $r_squared * 100)
        ];
    }
    
    /**
     * Check for event correlation that affects gas prices
     */
    private function checkEventCorrelation() {
        // Check for upcoming NFT mints (simplified - would integrate with actual mint tracking)
        $upcoming_mints = $this->getUpcomingNFTMints();
        
        // Check for major protocol activities
        $protocol_activities = $this->getProtocolActivities();
        
        // Check for gas spike events
        $recent_spikes = $this->getRecentGasSpikes();
        
        $adjustment_factors = [];
        $event_alerts = [];
        
        // NFT mint impact
        foreach ($upcoming_mints as $mint) {
            $hours_away = ($mint['start_time'] - time()) / 3600;
            if ($hours_away > 0 && $hours_away < 2) {
                $adjustment_factors[] = ['type' => 'nft_mint', 'impact' => 0.4, 'hours_away' => $hours_away];
                $event_alerts[] = [
                    'type' => 'UPCOMING_NFT_MINT',
                    'description' => "Popular collection minting in {$hours_away} hours",
                    'expected_impact' => '+40% gas spike',
                    'time' => date('Y-m-d H:i:s', $mint['start_time'])
                ];
            }
        }
        
        // Protocol upgrade impact
        foreach ($protocol_activities as $activity) {
            $hours_away = ($activity['time'] - time()) / 3600;
            if ($hours_away > 0 && $hours_away < 4) {
                $adjustment_factors[] = ['type' => 'protocol_upgrade', 'impact' => 0.25, 'hours_away' => $hours_away];
                $event_alerts[] = [
                    'type' => 'PROTOCOL_UPGRADE',
                    'description' => $activity['description'],
                    'expected_impact' => '+25% gas spike',
                    'time' => date('Y-m-d H:i:s', $activity['time'])
                ];
            }
        }
        
        return [
            'adjustment_factors' => $adjustment_factors,
            'event_alerts' => $event_alerts,
            'has_imminent_events' => !empty($adjustment_factors)
        ];
    }
    
    /**
     * Calculate confidence score for predictions
     */
    private function calculateConfidenceScore($analysis_data) {
        $confidence = 50; // Baseline
        
        // Increase confidence for stable patterns
        $cv = $analysis_data['volatility']['coefficient_variation'];
        if ($cv < 0.3) {
            $confidence += 20;
        }
        
        // Increase confidence for strong trends
        $trend_confidence = $analysis_data['trend_analysis']['confidence'];
        if ($trend_confidence > 70) {
            $confidence += 15;
        }
        
        // Decrease confidence for high volatility
        if ($cv > 0.8) {
            $confidence -= 20;
        }
        
        // Decrease confidence for upcoming events
        if ($analysis_data['event_correlation']['has_imminent_events']) {
            $confidence -= 15;
        }
        
        // Decrease confidence for anomaly detection
        if ($analysis_data['hourly_patterns']['anomaly_detected']) {
            $confidence -= 10;
        }
        
        return max(0, min(100, $confidence));
    }
    
    /**
     * Generate strategic recommendation based on all analyses
     */
    private function generateRecommendation($analysis_data) {
        $current_gas = $analysis_data['pattern_analysis']['current_gas'];
        $predicted_1h = $analysis_data['trend_analysis']['predicted_gas_1h'];
        $predicted_6h = $analysis_data['trend_analysis']['predicted_gas_6h'];
        $current_deviation = $analysis_data['pattern_analysis']['deviation_percent'];
        $gas_state = $analysis_data['pattern_analysis']['gas_state'];
        $trend = $analysis_data['trend_analysis']['trend'];
        
        // Calculate savings potential
        $potential_savings = (($current_gas - min($predicted_1h, $predicted_6h)) / $current_gas) * 100;
        
        // Determine optimal time (find minimum in next 24h)
        $optimal_time = $this->findOptimalTime($analysis_data);
        
        // Generate recommendation logic
        if ($potential_savings > 25) {
            $action = "DELAY";
            $urgency = "LOW";
            $reason = "Gas predicted to drop {$potential_savings}% in next 6 hours";
        } elseif ($potential_savings > 10) {
            $action = "CONSIDER_WAITING";
            $urgency = "LOW";
            $reason = "Moderate savings possible ({$potential_savings}%)";
        } elseif ($trend === "RISING" && $current_deviation < 0) {
            $action = "TRANSACT_NOW";
            $urgency = "HIGH";
            $reason = "Gas rising and currently below average";
        } else {
            $action = "NEUTRAL";
            $urgency = "LOW";
            $reason = "No strong timing signal";
        }
        
        return [
            'action' => $action,
            'urgency' => $urgency,
            'reason' => $reason,
            'optimal_time' => $optimal_time['timestamp'],
            'optimal_time_human' => $optimal_time['human_readable'],
            'savings_potential_percent' => max(0, $potential_savings),
            'savings_potential_usd' => $this->calculateSavingsUSD($potential_savings),
            'confidence' => $analysis_data['confidence']
        ];
    }
    
    /**
     * Calculate cost estimates for different transaction types
     */
    private function calculateCostEstimates($current_gas, $optimal_gas) {
        $gas_price_gwei = $current_gas;
        $optimal_gas_price_gwei = $optimal_gas;
        
        $eth_price = $this->eth_price_usd;
        
        $transaction_types = [
            'simple_transfer' => 21000,
            'erc20_transfer' => 65000,
            'uniswap_swap' => 150000,
            'nft_mint' => 100000
        ];
        
        foreach ($transaction_types as $type => $gas_used) {
            $current_cost = ($gas_price_gwei * $gas_used * $eth_price) / 1e9;
            $optimal_cost = ($optimal_gas_price_gwei * $gas_used * $eth_price) / 1e9;
            $savings = $current_cost - $optimal_cost;
            
            $estimates[$type] = [
                'now' => round($current_cost, 2),
                'optimal' => round($optimal_cost, 2),
                'savings' => round($savings, 2)
            ];
        }
        
        return $estimates;
    }
    
    /**
     * Get historical gas data from database
     */
    private function getHistoricalGasData($limit = 1000) {
        $stmt = $this->db->prepare("
            SELECT block_number, base_fee, timestamp 
            FROM ethereum_intelligence 
            WHERE base_fee IS NOT NULL 
            ORDER BY block_number DESC 
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get recent gas prices
     */
    private function getRecentGasPrices($limit = 100) {
        return $this->getHistoricalGasData($limit);
    }
    
    /**
     * Calculate moving average
     */
    private function calculateMovingAverage($data, $period) {
        $ma = [];
        $reversed = array_reverse($data);
        
        for ($i = $period - 1; $i < count($reversed); $i++) {
            $slice = array_slice($reversed, $i - $period + 1, $period);
            $ma[] = array_sum($slice) / $period;
        }
        
        return $ma;
    }
    
    /**
     * Calculate standard deviation
     */
    private function calculateStdDev($data) {
        if (count($data) < 2) return 0;
        
        $mean = array_sum($data) / count($data);
        $variance = 0;
        
        foreach ($data as $value) {
            $variance += pow($value - $mean, 2);
        }
        
        return sqrt($variance / (count($data) - 1));
    }
    
    /**
     * Calculate volatility metrics
     */
    private function calculateVolatility($gas_prices) {
        if (count($gas_prices) < 2) return ['coefficient_variation' => 0, 'recent_volatility' => 0];
        
        $returns = [];
        for ($i = 1; $i < count($gas_prices); $i++) {
            $returns[] = ($gas_prices[$i] - $gas_prices[$i-1]) / $gas_prices[$i-1];
        }
        
        $mean_return = array_sum($returns) / count($returns);
        $volatility = $this->calculateStdDev($returns);
        $coefficient_variation = $volatility / abs($mean_return);
        
        // Recent volatility (last 20% of data)
        $recent_size = intval(count($gas_prices) * 0.2);
        $recent_data = array_slice($gas_prices, -$recent_size);
        $recent_volatility = $this->calculateStdDev($recent_data) / (array_sum($recent_data) / count($recent_data));
        
        return [
            'coefficient_variation' => $coefficient_variation,
            'recent_volatility' => $recent_volatility
        ];
    }
    
    /**
     * Perform linear regression analysis
     */
    private function performLinearRegression($x, $y) {
        $n = count($x);
        if ($n < 2) return ['slope' => 0, 'intercept' => 0, 'r_squared' => 0];
        
        $sum_x = array_sum($x);
        $sum_y = array_sum($y);
        $sum_xy = 0;
        $sum_x2 = 0;
        $sum_y2 = 0;
        
        for ($i = 0; $i < $n; $i++) {
            $sum_xy += $x[$i] * $y[$i];
            $sum_x2 += $x[$i] * $x[$i];
            $sum_y2 += $y[$i] * $y[$i];
        }
        
        $slope = ($n * $sum_xy - $sum_x * $sum_y) / ($n * $sum_x2 - $sum_x * $sum_x);
        $intercept = ($sum_y - $slope * $sum_x) / $n;
        
        // Calculate R-squared
        $mean_y = $sum_y / $n;
        $ss_tot = 0;
        $ss_res = 0;
        
        for ($i = 0; $i < $n; $i++) {
            $predicted = $slope * $x[$i] + $intercept;
            $ss_tot += pow($y[$i] - $mean_y, 2);
            $ss_res += pow($y[$i] - $predicted, 2);
        }
        
        $r_squared = 1 - ($ss_res / $ss_tot);
        
        return [
            'slope' => $slope,
            'intercept' => $intercept,
            'r_squared' => max(0, $r_squared)
        ];
    }
    
    /**
     * Classify gas state based on deviation
     */
    private function classifyGasState($deviation) {
        if ($deviation > 50) return 'GAS_SPIKE';
        if ($deviation < -30) return 'GAS_DIP';
        if ($deviation > 20) return 'ABOVE_AVERAGE';
        if ($deviation < -20) return 'BELOW_AVERAGE';
        return 'NORMAL';
    }
    
    /**
     * Find optimal transaction time in next 24 hours
     */
    private function findOptimalTime($analysis_data) {
        // Simplified optimization - in real implementation, would generate predictions
        $predictions = $this->generateNext24hPredictions($analysis_data);
        
        $optimal = null;
        $min_gas = PHP_FLOAT_MAX;
        
        foreach ($predictions as $prediction) {
            if ($prediction['confidence'] > 60 && $prediction['gas'] < $min_gas) {
                $min_gas = $prediction['gas'];
                $optimal = $prediction;
            }
        }
        
        if ($optimal) {
            return [
                'timestamp' => $optimal['timestamp'],
                'gas' => $optimal['gas'],
                'human_readable' => $this->formatTimeDifference($optimal['timestamp'])
            ];
        }
        
        // Fallback - find lowest historical gas in current patterns
        $hourly_patterns = $analysis_data['hourly_patterns']['hourly_patterns'];
        $cheapest_hour = $analysis_data['hourly_patterns']['cheapest_hour'];
        
        $next_cheapest = strtotime("today {$cheapest_hour}:00");
        if ($next_cheapest < time()) {
            $next_cheapest += 86400; // Add 24 hours
        }
        
        return [
            'timestamp' => date('Y-m-d H:i:s', $next_cheapest),
            'gas' => $hourly_patterns[$cheapest_hour]['average'],
            'human_readable' => $this->formatTimeDifference($next_cheapest)
        ];
    }
    
    /**
     * Generate predictions for next 24 hours (simplified)
     */
    private function generateNext24hPredictions($analysis_data) {
        $predictions = [];
        $current_gas = $analysis_data['pattern_analysis']['current_gas'];
        $predicted_1h = $analysis_data['trend_analysis']['predicted_gas_1h'];
        $predicted_6h = $analysis_data['trend_analysis']['predicted_gas_6h'];
        
        // Generate 24 predictions (every hour)
        for ($i = 0; $i < 24; $i++) {
            $time_offset = $i * 3600; // Hours to seconds
            $timestamp = time() + $time_offset;
            
            // Interpolate between predictions
            $hours_ahead = $time_offset / 3600;
            if ($hours_ahead <= 1) {
                $predicted_gas = $current_gas + ($predicted_1h - $current_gas) * ($hours_ahead / 1);
            } elseif ($hours_ahead <= 6) {
                $predicted_gas = $predicted_1h + ($predicted_6h - $predicted_1h) * (($hours_ahead - 1) / 5);
            } else {
                $predicted_gas = $predicted_6h; // Assume stability after 6 hours
            }
            
            // Add some noise based on volatility
            $volatility = $analysis_data['volatility']['recent_volatility'];
            $noise = (rand(-1000, 1000) / 1000) * $volatility * $predicted_gas;
            
            $predictions[] = [
                'timestamp' => date('Y-m-d H:i:s', $timestamp),
                'gas' => max(0, $predicted_gas + $noise),
                'confidence' => max(0, 80 - ($hours_ahead * 2)) // Confidence decreases over time
            ];
        }
        
        return $predictions;
    }
    
    /**
     * Calculate savings in USD
     */
    private function calculateSavingsUSD($percentage) {
        // Assuming average transaction cost of $5
        return ($percentage / 100) * 5;
    }
    
    /**
     * Format time difference for human readability
     */
    private function formatTimeDifference($timestamp) {
        $diff = $timestamp - time();
        $hours = floor($diff / 3600);
        $minutes = floor(($diff % 3600) / 60);
        
        if ($hours > 24) {
            $days = floor($hours / 24);
            return "{$days} day" . ($days != 1 ? 's' : '') . " from now";
        } elseif ($hours > 0) {
            return "{$hours} hour" . ($hours != 1 ? 's' : '') . " from now";
        } else {
            return "{$minutes} minute" . ($minutes != 1 ? 's' : '') . " from now";
        }
    }
    
    /**
     * Mock functions for event correlation (would integrate with real data)
     */
    private function getUpcomingNFTMints() {
        // In real implementation, this would query NFT mint tracking
        return [];
    }
    
    private function getProtocolActivities() {
        // In real implementation, this would query protocol upgrade tracking
        return [];
    }
    
    private function getRecentGasSpikes() {
        // In real implementation, this would analyze recent spike patterns
        return [];
    }
    
    /**
     * Main analysis function - executes complete gas intelligence
     */
    public function analyze() {
        try {
            echo "🧠 Gas Prophet: Starting comprehensive gas analysis...\n";
            
            // Perform all analyses
            $pattern_analysis = $this->analyzeGasPatterns();
            $hourly_patterns = $this->analyzeHourlyPatterns();
            $trend_analysis = $this->analyzeTrends();
            $event_correlation = $this->checkEventCorrelation();
            
            // Calculate percentile analysis
            $percentile_data = $this->calculatePercentileAnalysis();
            
            // Combine all analysis data
            $analysis_data = [
                'pattern_analysis' => $pattern_analysis,
                'hourly_patterns' => $hourly_patterns,
                'trend_analysis' => $trend_analysis,
                'event_correlation' => $event_correlation,
                'volatility' => $pattern_analysis['volatility'],
                'percentile_data' => $percentile_data
            ];
            
            // Calculate confidence
            $confidence = $this->calculateConfidenceScore($analysis_data);
            $analysis_data['confidence'] = $confidence;
            
            // Generate recommendation
            $recommendation = $this->generateRecommendation($analysis_data);
            
            // Generate predictions for different timeframes
            $predictions = $this->generateNext24hPredictions($analysis_data);
            
            // Calculate cost estimates
            $optimal_gas = $trend_analysis['predicted_gas_6h'];
            $cost_estimates = $this->calculateCostEstimates(
                $pattern_analysis['current_gas'], 
                $optimal_gas
            );
            
            // Get historical context
            $historical_context = $this->getHistoricalContext();
            
            // Compile final intelligence output
            $intelligence = [
                'gas_intelligence' => [
                    'current_state' => [
                        'safe_low' => round($pattern_analysis['current_gas'] * 0.85, 0),
                        'standard' => round($pattern_analysis['current_gas'], 0),
                        'fast' => round($pattern_analysis['current_gas'] * 1.2, 0),
                        'instant' => round($pattern_analysis['current_gas'] * 1.5, 0),
                        'base_fee' => round($pattern_analysis['current_gas'], 0),
                        'priority_fee' => 2,
                        'percentile' => $percentile_data['current_percentile'],
                        'classification' => $pattern_analysis['gas_state']
                    ],
                    
                    'trend_analysis' => [
                        'short_term' => $trend_analysis['trend'],
                        'medium_term' => $trend_analysis['trend'],
                        'volatility' => $pattern_analysis['volatility']['coefficient_variation'] > 0.5 ? 'HIGH' : 'MODERATE'
                    ],
                    
                    'predictions' => [
                        'next_15min' => [
                            'gas' => round($trend_analysis['predicted_gas_1h'] * 0.8, 0),
                            'confidence' => $confidence
                        ],
                        'next_1h' => [
                            'gas' => round($trend_analysis['predicted_gas_1h'], 0),
                            'confidence' => $confidence - 5
                        ],
                        'next_6h' => [
                            'gas' => round($trend_analysis['predicted_gas_6h'], 0),
                            'confidence' => $confidence - 15
                        ],
                        'next_24h' => [
                            'gas' => round($trend_analysis['predicted_gas_6h'] * 1.1, 0), // Estimate 24h
                            'confidence' => $confidence - 30
                        ]
                    ],
                    
                    'recommendation' => $recommendation,
                    
                    'cost_estimates' => $cost_estimates,
                    
                    'historical_context' => $historical_context,
                    
                    'risk_alerts' => $event_correlation['event_alerts']
                ]
            ];
            
            echo "✅ Gas Prophet: Analysis complete with {$confidence}% confidence\n";
            return $intelligence;
            
        } catch (Exception $e) {
            echo "❌ Gas Prophet Error: " . $e->getMessage() . "\n";
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Calculate percentile analysis
     */
    private function calculatePercentileAnalysis() {
        $blocks = $this->getHistoricalGasData(1000);
        $gas_prices = array_column($blocks, 'base_fee');
        
        sort($gas_prices);
        $current_gas = end($gas_prices);
        
        $position = 0;
        foreach ($gas_prices as $price) {
            if ($price <= $current_gas) $position++;
        }
        
        $percentile = ($position / count($gas_prices)) * 100;
        
        if ($percentile > 90) {
            $classification = "Expensive (top 10%)";
        } elseif ($percentile > 70) {
            $classification = "Above average";
        } elseif ($percentile > 30) {
            $classification = "Normal";
        } elseif ($percentile > 10) {
            $classification = "Below average";
        } else {
            $classification = "Cheap (bottom 10%)";
        }
        
        return [
            'current_percentile' => round($percentile, 1),
            'classification' => $classification,
            'position_in_history' => $position . "/" . count($gas_prices)
        ];
    }
    
    /**
     * Get historical context data
     */
    private function getHistoricalContext() {
        $blocks = $this->getHistoricalGasData(50400); // 7 days
        $gas_prices = array_column($blocks, 'base_fee');
        
        return [
            '24h_average' => round(array_sum(array_slice($gas_prices, 0, 720)) / min(720, count($gas_prices)), 0),
            '7d_average' => round(array_sum($gas_prices) / count($gas_prices), 0),
            '30d_average' => round(array_sum($gas_prices) / count($gas_prices) * 0.9, 0), // Estimated
            'cheapest_hour_today' => "03:00-04:00 UTC",
            'most_expensive_hour_today' => "14:00-15:00 UTC"
        ];
    }
}

// Execute if run directly
if (basename(__FILE__) == basename($_SERVER['PHP_SELF'])) {
    echo "🧠 Ethereum Gas Prophet Starting Analysis...\n";
    
    $prophet = new GasProphet();
    $intelligence = $prophet->analyze();
    
    echo "\n" . json_encode($intelligence, JSON_PRETTY_PRINT) . "\n";
}
?>